//------------------------------------------------------------------------------
// File: cmn_slgameobj.cs
// Contains common algorithms to be performed on game objects
// Author: Matthew Rudge
//------------------------------------------------------------------------------

//-Object Globals---------------------------------------------------------------
$OST_BANDIT = 2;

// Train globals
$TT_NOTRAIN = -1;
$TT_WAGON1  = 0; //!< Wagon1 (appears when an outpost if upgraded)
$TT_WAGON2  = 1; //!< Wagon2 (appears when a train station is upgraded)
$TT_DEFAULT = 2; //!< Train that delivers nothing
$TT_WOOD    = 3; //!< Train that delivers wood
$TT_GOLD    = 4; //!< Train that delivers gold
$TT_FOOD    = 5; //!< Train that delivers food
$TT_COUNT   = 6; //!< The number of train types
//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
//! Indicates if the game object is employed somewhere
//! \param %obj Game object id
//! \retval bool True if the game object is a character and employed, false 
//! otherwise
////////////////////////////////////////////////////////////////////////////////
function slgIsEmployed(%obj)
{
   // Not a character
   if(!slgIsCharacter(%obj)) {
      return false;
   }
   
   // Get character component
   %cmpChar = slgQueryInterface(%obj, $CID_CHARACTER);
   return (%cmpChar.hasWork());
}

////////////////////////////////////////////////////////////////////////////////
//! Checks to see if game object is on a team
//! \param %team Team being queried
//! \retval bool True if the game object is on the team, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::isOnTeam(%this, %team)
{
   return (%this.getTeam() == %team);
}

////////////////////////////////////////////////////////////////////////////////
//! Checks to see if game object is of type
//! \param %type Type being queried
//! \retval bool True if the game object is of type, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::isOfType(%this, %type)
{
   return (%this.getDatablock().getName() $= %type);
}

////////////////////////////////////////////////////////////////////////////////
//! Test if game object is a building and has employees
//! \retval bool True if the game object is a building and has employees
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::hasEmployees(%this)
{
   %cmpBldg = slgQueryInterface(%this, $CID_BUILDING);
   if(!isObject(%cmpBldg)) {
      return false;
   }
   
   return (%cmpBldg.getEmployeeCount() > 0);
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the game object type
//! \retval string Game object type
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::getType(%this)
{
   return (%this.getDatablock().getName());
}

////////////////////////////////////////////////////////////////////////////////
//! Callback when an object has been selected
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::onSelect(%this)
{
   // Character object
   if(slgIsCharacter(%this))
   {
      %cmpAI = slgQueryInterface(%this, $CID_AI);
      if (IsFollowingPath(%cmpAI) && %cmpAI.hasAction() == false)
      {
         StopMoving(%cmpAI);
         %this.playThread(0, "root");
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Returns true or false if object is a luxury building
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::isLuxuryBuilding(%object)
{
   switch$(%object.getType()) {
      case "Gazebo":
         return true;
      case "Church":
         return true;
      case "Barbershop":
         return true;
      case "Outhouse":
         return true;
      case "welcomesign":
         return true;
      case "Fountain":
         return true;
      case "Statue":
         return true;
      case "Townbell":
         return true;
      case "Flowerpot":
         return true;
      case "Cemetery":
         return true;
      case "Bridge":
         return true;
      case "Roadway":
         return true;
      default:
         return false;
   }
}

function SLGameObj::isTownBuilding(%object)
{
   switch$(%object.getType())
   {
      case "Honeypot":
         return false;
      case "Bridge":
         return false;
   }
   
   return !%object.isLuxuryBuilding();
}

////////////////////////////////////////////////////////////////////////////////
//! Calculates the squared distance to an object
//! \param %object Object to calculate squared distance to. Returns -1 for 
//! failure
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::sqDistanceTo(%this, %object)
{
   if(!isObject(%object)) {
      return -1;
   }
   
   %posThis = %this.getPosition();
   %xThis   = getWord(%posThis, 0);
   %yThis   = getWord(%posThis, 1);
   %zThis   = getWord(%posThis, 2);
   
   %posObj  = %object.getPosition();
   %xObj    = getWord(%posObj, 0);
   %yObj    = getWord(%posObj, 1);
   %zObj    = getWord(%posObj, 2);
   
   // Calculate squared distance
   %dx = %xThis - %xObj;
   %dy = %yThis - %yObj;
   %dz = %zThis - %zObj;
   return ((%dx * %dx) + (%dy * %dy) + (%dz * %dz));
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the server id of the object
//! \retval Server id
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::getServerId(%this)
{
   %ghost     = ServerConnection.getGhostID(%this);
   %serverObj = ClientGroup.getObject(0).resolveObjectFromGhostIndex(%ghost);
   if(isObject(%serverObj)) {
      return %serverObj;
   }
   else {
      return %this;
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the client id of the object
//! \retval Client id
////////////////////////////////////////////////////////////////////////////////
function SLGameObj::getClientId(%this)
{ 
   %ghost     = ClientGroup.getObject(0).getGhostID(%this);
   %clientObj = ServerConnection.resolveGhost(%ghost);
   if(isObject(%clientObj)) {
      return %clientObj;
   }
   else {
      return %this;
   }
}

function CCmpBuilding::getCurrentTaxAmount(%this, %characterList)
{
   if(!isObject(%characterList)) {
      return 0;
   }
   if(%this.currentTax $= "") {
      return 0;
   }
   
   // Tax everyone employed
   %charCnt = 0;
   %charListSize = %characterList.getSize();
   for (%i = 0; %i < %charListSize; %i++)
   {
      %objChar = %characterList.getID(%i);
      if(!%objChar.isOnTeam($OST_PLAYER)) {
         continue;
      }
      
      // Get character component
      %cmpChar = slgQueryInterface(%characterList.getID(%i), $CID_CHARACTER);
      if(isObject(%cmpChar)) {
         // If character is employed then increment count
         if(%cmpChar.hasWork()) {
            %charCnt++;
         }
      }
   }
   return (%charCnt * %this.currentTax);
}

// End cmn_slgameobj.cs